<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Mail\VerificationEmail;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;

class UserController extends Controller
{
    public function index()
    {
        $users = User::all();
        return view('users.index', compact('users'));
    }

    public function create()
    {
        return view('users.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'username' => 'required|string|unique:users',
            'email' => 'required|email|unique:users',
            'password' => 'required|confirmed|min:5',
            'akses' => 'required|in:admin,sales',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $photoPath = 'images/default.jpg';

        if ($request->hasFile('photo')) {
            $photoPath = $request->file('photo')->store('photos', 'public');
            $photoPath = 'storage/' . $photoPath;
        }

        $verificationToken = Str::random(64);

        $user = User::create([
            'username' => $request->username,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'akses' => $request->akses,
            'photo' => $photoPath,
            'verification_token' => $verificationToken,
        ]);

        // Kirim email verifikasi
        Mail::to($user->email)->send(new VerificationEmail($user));

        return redirect()->route('users.index')->with('success', 'User berhasil ditambahkan. Email verifikasi telah dikirim.');
    }

    public function verifyEmail($token)
    {
        $user = User::where('verification_token', $token)->first();

        if (!$user) {
            return redirect('/login')->with('error', 'Token verifikasi tidak valid.');
        }

        $user->email_verified_at = now();
        $user->verification_token = null;
        $user->save();

        return redirect('/login')->with('success', 'Email berhasil diverifikasi. Silakan login.');
    }

    public function edit($id)
    {
        $user = User::findOrFail($id);
        return view('users.edit', compact('user'));
    }

    public function update(Request $request, $id)
    {
        $user = User::findOrFail($id);

        $request->validate([
            'username' => 'required|string|unique:users,username,' . $user->id,
            'email' => 'required|email|unique:users,email,' . $user->id,
            'akses' => 'required|in:admin,sales',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $photoPath = $user->photo;

        if ($request->hasFile('photo')) {
            if ($user->photo && $user->photo != 'images/default.jpg' && Storage::disk('public')->exists(str_replace('storage/', '', $user->photo))) {
                Storage::disk('public')->delete(str_replace('storage/', '', $user->photo));
            }

            $photoPath = $request->file('photo')->store('photos', 'public');
            $photoPath = 'storage/' . $photoPath;
        }

        $user->update([
            'username' => $request->username,
            'email' => $request->email,
            'akses' => $request->akses,
            'photo' => $photoPath,
        ]);

        return redirect()->route('users.index')->with('success', 'User berhasil diperbarui.');
    }

    public function destroy($id)
    {
        $user = User::findOrFail($id);

        if ($user->photo && $user->photo != 'images/default.jpg' && Storage::disk('public')->exists(str_replace('storage/', '', $user->photo))) {
            Storage::disk('public')->delete(str_replace('storage/', '', $user->photo));
        }

        $user->delete();

        return redirect()->route('users.index')->with('success', 'User berhasil dihapus.');
    }
}
