<?php

namespace App\Http\Controllers;

use App\Models\Stok;
use App\Models\Mitra;
use App\Models\Produk;
use App\Exports\StoksExport;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Excel;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Foundation\Auth\User;
use Illuminate\Support\Facades\Auth;

class StokController extends Controller
{
    public function index()
    {
        $stoks = Stok::with(['produk', 'mitra'])
            ->latest()
            ->paginate(10);

        return view('stoks.index', compact('stoks'));
    }

    public function create(Request $request)
    {
        $produks = Produk::all();
        $mitras = Mitra::with('user')->get(); // ambil mitra beserta penanggung jawab
        // Ambil semua user untuk dropdown penanggung jawab
        $users = User::all();

        $produk_id = $request->get('idproduk');
        $mitra_id = $request->get('mitra_id');

        // Hitung total sisa stok untuk produk dan mitra yang dipilih, default 0 jika belum ada
        $sisa = 0;
        if ($produk_id && $mitra_id) {
            $sisa = DB::table('stoks')
                ->where('produk_id', $produk_id)
                ->where('mitra_id', $mitra_id)
                ->sum(DB::raw('masuk - terjual + `return`'));
        }

        return view('stoks.create', compact('produks', 'mitras', 'sisa', 'mitra_id', 'users'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'produk_id' => 'required|exists:produks,id',
            'mitra_id' => 'required|exists:mitras,id',
            'user_id' => 'required|exists:users,id',
            'date' => 'required|date',
            'masuk' => 'required|numeric|min:0',
            'terjual' => 'required|numeric|min:0',
            'return' => 'required|numeric|min:0',
            'sisa' => 'required|numeric|min:0',
        ]);

        // Update penanggung jawab mitra
        $mitra = Mitra::findOrFail($validated['mitra_id']);
        $mitra->user_id = $validated['user_id'];
        $mitra->save();

        // Simpan stok
        Stok::create([
            'produk_id' => $validated['produk_id'],
            'mitra_id' => $validated['mitra_id'],
            'date' => $validated['date'],
            'masuk' => $validated['masuk'],
            'terjual' => $validated['terjual'],
            'return' => $validated['return'],
            'sisa' => $validated['sisa'],
            'jenis' => 'penjualan',
        ]);

        return redirect()->route('mitras.show', $validated['mitra_id'])
            ->with('success', 'Data penjualan berhasil disimpan.');
    }



    public function show($id)
    {
        $stok = Stok::with(['produk', 'mitra'])->findOrFail($id);

        return view('stoks.show', compact('stok'));
    }

    public function edit($id)
    {
        $stok = Stok::findOrFail($id);
        $produks = Produk::orderBy('name')->get();
        $mitras = Mitra::orderBy('name')->get();
        $users = User::orderBy('username')->get();
    
        // Hitung total stok tersisa dari produk ini di semua mitra
        $sisaTotalProduk = Stok::where('produk_id', $stok->produk_id)->sum('sisa');
    
        return view('stoks.edit', compact('stok', 'produks', 'mitras', 'users', 'sisaTotalProduk'));
    }
    


    public function update(Request $request, $id)
    {
        $request->validate([
            'date' => 'required|date',
            'produk_id' => 'required|exists:produks,id',
            'mitra_id' => 'required|exists:mitras,id',
            'user_id' => 'required|exists:users,id',
            'masuk' => 'required|integer|min:0',
            'terjual' => 'required|integer|min:0',
            'return' => 'required|integer|min:0',
        ]);

        
        // // Validasi tambahan
        // $totalStokTersedia = $request->masuk + $request->return;
        // if ($request->terjual > $totalStokTersedia) {
        //     return back()->withErrors([
        //         'terjual' => 'Jumlah terjual tidak boleh melebihi jumlah masuk + return.'
        //     ])->withInput();
        // }
        // // Validasi tambahan: return tidak boleh melebihi terjual
        // if ($request->return > $request->terjual) {
        //     return back()->withErrors([
        //         'return' => 'Jumlah return tidak boleh melebihi jumlah terjual.'
        //     ])->withInput();
        // }


        $stok = Stok::findOrFail($id);
        $produk = Produk::findOrFail($request->produk_id);

        // Update penanggung jawab mitra
        $mitra = Mitra::findOrFail($request->mitra_id);
        $mitra->user_id = $request->user_id;
        $mitra->save();

        // $sisa = $totalStokTersedia - $request->terjual;
        $totalHarga = $request->terjual * $produk->price;

        $stok->update([
            'date' => $request->date,
            'produk_id' => $request->produk_id,
            'mitra_id' => $request->mitra_id,
            'masuk' => $request->masuk,
            'terjual' => $request->terjual,
            'return' => $request->return,
            'sisa' => $sisa,
            'total_harga' => $totalHarga,
        ]);

        return redirect()->route('stoks.index')->with('success', 'Stok berhasil diperbarui.');
    }


    public function destroy($id)
    {
        $stok = Stok::findOrFail($id);
        $stok->delete();

        return redirect()->route('stoks.index')->with('success', 'Stok berhasil dihapus.');
    }

    //     public function export()
    // {
    //     return Excel::download(new StoksExport, 'laporan-stok.xlsx');
    // }
}
