<?php

namespace App\Http\Controllers;

use App\Models\mitra_produk;
use App\Models\Produk;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ProdukController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->input('search');
        
        // Gunakan where condition jika ada parameter pencarian
        $produks = Produk::when($search, function ($query, $search) {
            $query->where('name', 'like', "%{$search}%")
                  ->orWhere('explanation', 'like', "%{$search}%");
        })->latest()->paginate(6); // Gunakan paginate()

        return view('produks.index', compact('produks'));
    }

    public function create()
    {
        return view('produks.create');
    }

    public function store(Request $request)
    {
        // Validasi input dari form
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'explanation' => 'required|string',
            'price' => 'required|numeric',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        // Cek jika ada gambar yang diupload dan simpan
        if ($request->hasFile('image')) {
            $validated['image'] = $this->uploadImage($request->file('image'));
        }
          // Tambahkan ID pengguna yang sedang login
    $validated['user_id'] = auth()->id();

        Produk::create($validated); // Simpan produk ke database

        return redirect()->route('produks.index')->with('success', 'Produk berhasil ditambahkan.');
    }

    public function edit(Produk $produk)
    {
        return view('produks.edit', compact('produk'));
    }

    public function update(Request $request, Produk $produk)
    {
        // Validasi input dari form
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'explanation' => 'required|string',
            'price' => 'required|numeric',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        // Cek jika ada gambar baru yang diupload
        if ($request->hasFile('image')) {
            // Hapus gambar lama jika ada
            if ($produk->image && file_exists(storage_path('app/public/' . $produk->image))) {
                unlink(storage_path('app/public/' . $produk->image));
            }
            // Upload gambar baru
            $validated['image'] = $this->uploadImage($request->file('image'));
        }

        $produk->update($validated); // Update produk di database

        return redirect()->route('produks.index')->with('success', 'Produk berhasil diedit.');
    }

    public function destroy(Produk $produk)
    {
        // Hapus gambar jika ada
        if ($produk->image && file_exists(storage_path('app/public/' . $produk->image))) {
            unlink(storage_path('app/public/' . $produk->image));
        }

        $produk->delete(); // Hapus produk dari database

        return redirect()->route('produks.index')->with('success', 'Produk berhasil dihapus.');
    }

    public function show(Produk $produk)
    {
        return view('produks.show', compact('produk'));
    }

    /**
     * Fungsi untuk menangani upload gambar produk
     */
    private function uploadImage($image)
    {
        return $image->store('produk_images', 'public');
    }

    public function setProduk(Request $request)
{
    // Validasi
    $request->validate([
        'mitra_id' => 'required|exists:mitras,id',
        'produk_ids' => 'array',
        'produk_ids.*' => 'exists:produks,id',
    ]);

    $mitraId = $request->mitra_id;
    $produkIds = $request->produk_ids ?? [];

    // Ambil semua produk yang sudah dimiliki mitra dari pivot table
    $produkTersimpan = DB::table('mitra_produks')
        ->where('mitra_id', $mitraId)
        ->pluck('produk_id')
        ->toArray();

    // Tambah produk yang baru dicentang (belum ada)
    foreach ($produkIds as $produkId) {
        if (!in_array($produkId, $produkTersimpan)) {
            DB::table('mitra_produks')->insert([
                'mitra_id' => $mitraId,
                'produk_id' => $produkId,
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        }
    }

    // Hapus produk yang sebelumnya ada tapi sekarang tidak dicentang
    foreach ($produkTersimpan as $oldProdukId) {
        if (!in_array($oldProdukId, $produkIds)) {
            DB::table('mitra_produks')
                ->where('mitra_id', $mitraId)
                ->where('produk_id', $oldProdukId)
                ->delete();
        }
    }

    return redirect()->back()->with('berhasil', 'Produk mitra berhasil diperbarui!');
}
}
