<?php

namespace App\Http\Controllers;

use App\Models\Stok;
use App\Models\User;
use App\Models\Mitra;
use App\Models\Produk;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class MitraController extends Controller
{
    public function index()
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        if ($user->isAdmin()) {
            $mitras = Mitra::all();
            $produks = Produk::All();
        } elseif ($user->isSales()) {
            $mitras = Mitra::all();
            $produks = Produk::All();
        } else {
            $mitras = collect();
        }

        return view('mitras.index', compact('mitras', 'produks'));
    }

    public function create()
    {
        $sales = User::where('akses', 'sales')->get();
        return view('mitras.create', compact('sales'));
    }
    public function store(Request $request)
    {
        // Validasi input
        $data = $request->validate([
            'name' => 'required|string|max:255',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'latitude' => 'required|numeric',
            'longitude' => 'required|numeric',
            // 'user_id' => 'required|exists:users,id', // tambahkan validasi user_id
        ]);

        // Tambahkan user_id (gunakan penugasan, bukan perbandingan)
        $data['user_id'] = Auth::id();

        // Simpan gambar jika ada
        if ($request->hasFile('image')) {
            $path = $request->file('image')->store('mitras', 'public');
            $data['image'] = $path;
        }

        // Simpan ke database
        Mitra::create($data);

        // Redirect dengan pesan sukses
        return redirect()->route('mitras.index')->with('success', 'Mitra berhasil ditambahkan!');
    }


    public function show($id)
    {
        $mitra = Mitra::with(['produks', 'user'])->findOrFail($id);

        // Hitung total sisa per produk untuk mitra ini
        $stokPerProduk = [];

        foreach ($mitra->produks as $produk) {
            $totalSisa = Stok::where('produk_id', $produk->id)
                ->where('mitra_id', $mitra->id)
                ->sum('sisa');

            $stokPerProduk[$produk->id] = $totalSisa;
        }

        return view('mitras.show', compact('mitra', 'stokPerProduk'));
    }


    public function edit(Mitra $mitra)
    {
        $sales = User::where('akses', 'sales')->get();
        $uploadedImage = session('uploaded_image_edit_' . $mitra->id, null);
        return view('mitras.edit', compact('mitra', 'sales', 'uploadedImage'));
    }

    public function update(Request $request, $id)
    {
        $mitra = Mitra::findOrFail($id);
        $sessionKey = 'uploaded_image_edit_' . $mitra->id;

        // Kalau ada file baru diupload
        if ($request->hasFile('image')) {
            $request->validate([
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            ]);

            // Hapus file lama di storage jika ada
            if ($mitra->image && Storage::disk('public')->exists($mitra->image)) {
                Storage::disk('public')->delete($mitra->image);
            }

            $path = $request->file('image')->store('mitras', 'public');
            $request->session()->put($sessionKey, $path);
        }

        // Validasi form selain file
        $data = $request->validate([
            'name' => 'required|string|max:255',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
            // 'user_id' => 'required|exists:users,id',
        ]);

        // Ambil image path dari session (kalau ada file baru)
        $data['image'] = $request->session()->get($sessionKey, $mitra->image);
        $data['user_id'] = $mitra->user_id;

        // Update DB
        $mitra->update($data);

        // Hapus session image
        $request->session()->forget($sessionKey);

        return redirect()->route('mitras.index')->with('success', 'Mitra berhasil diperbarui.');
    }



    public function destroy($id)
    {
        $mitra = Mitra::findOrFail($id);

        if ($mitra->image && Storage::disk('public')->exists($mitra->image)) {
            Storage::disk('public')->delete($mitra->image);
        }

        $mitra->delete();

        return redirect()->route('mitras.index')->with('success', 'Mitra berhasil dihapus!');
    }

    public function salesMitra()
    {
        $user = auth()->user();

        if ($user->akses !== 'sales') {
            abort(403, 'Tidak diizinkan.');
        }

        $mitras = Mitra::where('user_id', $user->id)->get();

        return view('mitras.sales', compact('mitras'));
    }
}
