<?php

namespace App\Http\Controllers;

use App\Models\Stok;
use App\Models\User;
use App\Models\Mitra;
use App\Models\Produk;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\PengirimanStokExport;
use Carbon\Carbon;

class LaporanController extends Controller
{
    public function index()
{
    // Semua bisa melihat data
    $stoks = Stok::with(['produk', 'mitra'])->latest()->get();

    return view('laporan.index', compact('stoks'));
}


    public function create(Request $request)
    {
        $produkId = $request->query('idproduk');
        $mitraId = $request->query('mitra_id');
        $salesId = $request->query('sales_id');

        $produk = Produk::findOrFail($produkId);
        $mitra = Mitra::findOrFail($mitraId);

        // Ambil 1 data stok terakhir berdasarkan produk & mitra
        $stokTerakhir = Stok::where('produk_id', $produkId)
            ->where('mitra_id', $mitraId)
            ->orderBy('date', 'desc')->orderBy('created_at', 'desc')
            ->first();

        $sisa = $stokTerakhir ? $stokTerakhir->sisa : 0;

        // Jika admin, ambil daftar sales untuk dropdown
        $users = null;
        $userTerpilih = null;
        if (Auth::user()->akses !== 'sales') {
            $users = User::where('akses', 'sales')->get();
            $userTerpilih = User::find($salesId);
        }

        return view('laporan.create', compact('produk', 'mitra', 'sisa', 'users', 'userTerpilih'));
    }



    public function store(Request $request)
    {
        $user = Auth::user();

        $data = $request->validate([
            'date' => 'required|date',
            'produk_id' => 'required|exists:produks,id',
            'mitra_id' => 'required|exists:mitras,id',
            'masuk' => 'required|numeric|min:0',
            'terjual' => 'required|numeric|min:0',
            'return' => 'required|numeric|min:0',
            'sisa' => 'required|numeric',
            'sales_id' => 'nullable|exists:users,id',
        ]);



        // Hitung harga
        $produk = Produk::findOrFail($data['produk_id']);
        $data['harga_satuan'] = $produk->price;
        $data['total_harga'] = $data['masuk'] * $produk->price;

        Stok::create($data);

        return redirect()->route('pengiriman.index')->with('success', 'Pengiriman berhasil ditambahkan!');
    }



    public function edit($id)
    {
        $pengiriman = Stok::findOrFail($id);
        $produks = Produk::all();
        $mitras = Mitra::all();
        $users = User::where('akses', 'sales')->get();

        // Ambil stok terakhir SEBELUM yang sedang diedit
        $sisaSebelumnya = Stok::where('produk_id', $pengiriman->produk_id)
            ->where('mitra_id', $pengiriman->mitra_id)
            ->where('id', '<', $pengiriman->id)
            ->orderBy('date', 'desc')
            ->orderBy('created_at', 'desc')
            ->first();

        $sisa = $sisaSebelumnya ? $sisaSebelumnya->sisa : 0;

        return view('laporan.edit', compact(
            'pengiriman',
            'produks',
            'mitras',
            'users',
            'sisa'
        ));
    }


    public function update(Request $request, $id)
    {
        $user = Auth::user();
        $stok = Stok::findOrFail($id);

        if ($user->akses === 'sales' && $stok->mitra->user_id !== $user->id) {
            abort(403, 'Unauthorized action.');
        }

        $data = $request->validate([
            'date' => 'required|date',
            'produk_id' => 'required|exists:produks,id',
            'mitra_id' => 'required|exists:mitras,id',
            'masuk' => 'required|numeric|min:0',
            'terjual' => 'required|numeric|min:0',
            'return' => 'required|numeric|min:0',
            'sisa' => 'required|numeric',
            'sales_id' => 'required|exists:users,id',
        ]);



        // Hitung harga
        $produk = Produk::findOrFail($data['produk_id']);
        $data['harga_satuan'] = $produk->price;
        $data['total_harga'] = $data['masuk'] * $produk->price;

        $stok->update($data);

        return redirect()->route('pengiriman.index')->with('success', 'Pengiriman berhasil diperbarui!');
    }

    public function destroy($id)
    {
        $user = Auth::user();
        $stok = Stok::findOrFail($id);

        if ($user->akses === 'sales' && $stok->mitra->user_id !== $user->id) {
            abort(403, 'Unauthorized action.');
        }

        $stok->delete();

        return redirect()->route('pengiriman.index')->with('success', 'Pengiriman berhasil dihapus!');
    }

    public function export()
    {
        return Excel::download(new PengirimanStokExport, 'laporan-pengiriman.xlsx');
    }
}
