<?php

namespace App\Http\Controllers;

use App\Models\Stok;
use App\Models\Mitra;
use App\Models\Produk;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;

class StokController extends Controller
{
    public function index()
    {
        $stoks = Stok::with(['produk', 'mitra'])
            ->latest()
            ->paginate(10);

        return view('stoks.index', compact('stoks'));
    }

    public function create()
    {
        $produks = Produk::orderBy('name')->get();
        $mitras = Mitra::orderBy('name')->get();

        return view('stoks.create', compact('produks', 'mitras'));
    }

    public function store(Request $request)
{
    $request->validate([
        'produk_id' => 'required|exists:produks,id',
        'mitra_id' => 'required|exists:mitras,id',
        'jumlah' => 'integer|min:1',
        'date' => 'required|date',
    ]);

    $produk = Produk::findOrFail($request->produk_id);
    $hargaSatuan = $produk->price; // pastikan produk punya harga
    $totalHarga = $request->jumlah * $hargaSatuan;

    Stok::create([
        'produk_id' => $request->produk_id,
        'mitra_id' => $request->mitra_id,
        'jumlah' => $request->jumlah,
        'date' => $request->date,
        'harga_satuan' => $hargaSatuan,
        'total_harga' => $totalHarga,
        'keterangan' => $request->keterangan,
    ]);

    return redirect()->route('stoks.index')->with('success', 'Stok berhasil ditambahkan dan data keuangan tercatat.');
}


    public function show($id)
    {
        $stok = Stok::with(['produk', 'mitra'])->findOrFail($id);

        return view('stoks.show', compact('stok'));
    }

    public function edit($id)
    {
        $stok = Stok::findOrFail($id);
        $produks = Produk::orderBy('name')->get();
        $mitras = Mitra::orderBy('name')->get();

        return view('stoks.edit', compact('stok', 'produks', 'mitras'));
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'date' => 'required|date',
            'produk_id' => 'required|exists:produks,id',
            'mitra_id' => 'required|exists:mitras,id',
            'masuk' => 'required|integer|min:0',
            'terjual' => 'required|integer|min:0',
            'return' => 'required|integer|min:0',
        ]);

        $stok = Stok::findOrFail($id);

        $produk = Produk::findOrFail($request->produk_id);
        $totalHarga = $request->masuk * $produk->harga;

        $sisa = $request->masuk - $request->terjual + $request->return;

        $stok->update([
            'date' => $request->date,
            'produk_id' => $request->produk_id,
            'mitra_id' => $request->mitra_id,
            'masuk' => $request->masuk,
            'terjual' => $request->terjual,
            'return' => $request->return,
            'sisa' => max($sisa, 0),
            'total_harga' => $totalHarga,
        ]);

        return redirect()->route('stoks.index')->with('success', 'Stok berhasil diperbarui.');
    }

    public function destroy($id)
    {
        $stok = Stok::findOrFail($id);
        $stok->delete();

        return redirect()->route('stoks.index')->with('success', 'Stok berhasil dihapus.');
    }
}
