<?php

namespace App\Http\Controllers;

use App\Models\Stok;
use App\Models\Mitra;
use App\Models\Produk;
use Illuminate\Http\Request;
use App\Models\PengirimanStok;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class PengirimanStokController extends Controller
{
    public function index()
    {
        $user = Auth::user();

        $pengirimanStoks = PengirimanStok::with(['mitra', 'produk'])
            ->when($user->akses !== 'admin', function ($query) use ($user) {
                $query->whereHas('mitra', fn ($q) => $q->where('user_id', $user->id));
            })
            ->latest()
            ->get();

        return view('pengiriman-stok.index', compact('pengirimanStoks'));
    }

    public function create()
    {
        return view('pengiriman-stok.create', [
            'mitras' => Mitra::all(),
            'produks' => Produk::all(),
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'produk_id'     => 'required|exists:produks,id',
            'mitra_id'      => 'required|exists:mitras,id',
            'tanggal_kirim' => 'required|date',
            'jumlah'        => 'required|integer|min:1',
        ]);

        $produk = Produk::findOrFail($validated['produk_id']);
        $harga_satuan = $produk->price ?? 0;

        if ($harga_satuan <= 0) {
            return back()->withErrors(['harga_satuan' => 'Harga produk belum ditentukan atau 0.'])->withInput();
        }

        $total_harga = $validated['jumlah'] * $harga_satuan;

        PengirimanStok::create([
            ...$validated,
            'status'        => 'pending',
            'harga_satuan'  => $harga_satuan,
            'total_harga'   => $total_harga,
            'user_id'       => Auth::id(),
        ]);

        return redirect()->route('pengiriman-stok.index')->with('success', 'Pengiriman stok berhasil dibuat.');
    }

    public function konfirmasi($id)
{
    $pengiriman = PengirimanStok::findOrFail($id);

    if ($pengiriman->status === 'diterima') {
        return back()->with('info', 'Pengiriman sudah dikonfirmasi sebelumnya.');
    }

    DB::transaction(function () use ($pengiriman) {
        $pengiriman->update([
            'status'         => 'diterima',
            'tanggal_terima' => now(),
        ]);

        $stok = Stok::where('mitra_id', $pengiriman->mitra_id)
            ->where('produk_id', $pengiriman->produk_id)
            ->first();

        if ($stok) {
            $stok->masuk        += $pengiriman->jumlah;
             $stok->jumlah     += $pengiriman->jumlah;
            $stok->sisa         += $pengiriman->jumlah;
            $stok->harga_satuan = $pengiriman->harga_satuan;
            $stok->total_harga  += $pengiriman->total_harga;
            $stok->save();
        } else {
            Stok::create([
                'produk_id'     => $pengiriman->produk_id,
                'mitra_id'      => $pengiriman->mitra_id,
                'date'          => now(),
                'masuk'         => $pengiriman->jumlah,
                'jumlah'      => $pengiriman->jumlah,
                'terjual'       => 0,
                'return'        => 0,
                'sisa'          => $pengiriman->jumlah,
                'sales_id'      => Auth::id(),
                'harga_satuan'  => $pengiriman->harga_satuan,
                'total_harga'   => $pengiriman->total_harga,
            ]);
        }
    });

    return back()->with('success', 'Pengiriman dikonfirmasi dan stok mitra diperbarui.');
}


public function update(Request $request, $id)
{
    $pengiriman = PengirimanStok::findOrFail($id);

    $validated = $request->validate([
        'produk_id'     => 'required|exists:produks,id',
        'mitra_id'      => 'required|exists:mitras,id',
        'tanggal_kirim' => 'required|date',
        'jumlah'        => 'required|integer|min:1',
        'status'        => 'nullable|string|in:pending,diterima,batal',
    ]);

    $produk = Produk::findOrFail($validated['produk_id']);
    $harga_satuan = $produk->harga ?? 0;
    $total_harga = $validated['jumlah'] * $harga_satuan;

    // Update pengiriman_stok
    $pengiriman->update([
        ...$validated,
        'harga_satuan' => $harga_satuan,
        'total_harga'  => $total_harga,
    ]);

    // Jika pengiriman sudah diterima, update data di tabel stok
    if ($pengiriman->status === 'diterima') {
        $stok = Stok::where('mitra_id', $pengiriman->mitra_id)
            ->where('produk_id', $pengiriman->produk_id)
            ->first();

        if ($stok) {
            // Hitung ulang nilai berdasarkan total pengiriman (bisa dikembangkan pakai akumulasi semua pengiriman)
            $stok->harga_satuan = $harga_satuan;
            $stok->total_harga  = $stok->total_harga - $pengiriman->total_harga + $total_harga;

            // Sesuaikan jumlah masuk & stok jika jumlah diubah
            $stok->masuk = $stok->masuk - $pengiriman->jumlah + $validated['jumlah'];
            $stok->jumlah = $stok->jumlah - $pengiriman->jumlah + $validated['jumlah'];
            $stok->sisa   = $stok->sisa   - $pengiriman->jumlah + $validated['jumlah'];

            $stok->save();
        }
    }

    return redirect()->route('pengiriman-stok.index')->with('success', 'Data pengiriman berhasil diperbarui.');
}

}
