<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Stok;
use App\Models\Mitra;
use App\Models\Produk;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;

class DashboardController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();

        // ADMIN DASHBOARD
        if ($user->akses === 'admin') {
            $totalProduk = Produk::count();
            $totalMitra = Mitra::count();
            $totalStok = Stok::sum('sisa');

            $tahunDipilih = $request->get('tahun');
            $daftarTahun = Stok::selectRaw('YEAR(date) as tahun')
                ->distinct()
                ->orderByDesc('tahun')
                ->pluck('tahun');

            $penjualanQuery = Stok::selectRaw('MONTH(date) as bulan, YEAR(date) as tahun, SUM(terjual) as total');
            if ($tahunDipilih) {
                $penjualanQuery->whereYear('date', $tahunDipilih);
            }

            $penjualanData = $penjualanQuery
                ->groupByRaw('YEAR(date), MONTH(date)')
                ->orderByRaw('YEAR(date), MONTH(date)')
                ->get();

            $labelBulan = [];
            $dataTerjual = [];

            foreach ($penjualanData as $data) {
                $label = date('M', mktime(0, 0, 0, $data->bulan, 1)) . ' ' . $data->tahun;
                $labelBulan[] = $label;
                $dataTerjual[] = (int) $data->total;
            }

            // Jika kosong, set default supaya chart tidak error
            if (empty($labelBulan)) {
                $labelBulan[] = 'Jan';
                $dataTerjual[] = 0;
            }

            // Menampilkan stok yang menipis berdasarkan akumulasi per produk-mitra
            $stokMenipis = Stok::join('produks', 'produks.id', '=', 'stoks.produk_id')
                ->select('stoks.produk_id', 'stoks.mitra_id')
                ->selectRaw('SUM(stoks.sisa) as sisa')
                ->addSelect('produks.minimum_stok')
                ->with(['produk', 'mitra'])
                ->groupBy('stoks.produk_id', 'stoks.mitra_id', 'produks.minimum_stok')
                ->havingRaw('SUM(stoks.sisa) <= MIN(produks.minimum_stok)')
                ->get();

            return view('dashboard.index', compact(
                'totalProduk',
                'totalMitra',
                'totalStok',
                'labelBulan',
                'dataTerjual',
                'daftarTahun',
                'tahunDipilih',
                'stokMenipis'
            ));
        }

        // SALES DASHBOARD
        if ($user->akses === 'sales') {
            $userId = $user->id;

            $totalMitraSales = Mitra::where('user_id', $userId)->count();

            // Stok menipis versi SALES (group by produk-mitra)
            $stokMenipisSales = Stok::join('produks', 'produks.id', '=', 'stoks.produk_id')
                ->join('mitras', 'mitras.id', '=', 'stoks.mitra_id')
                ->where('mitras.user_id', $userId)
                ->select('stoks.produk_id', 'stoks.mitra_id')
                ->selectRaw('SUM(stoks.sisa) as sisa')
                ->addSelect('produks.minimum_stok')
                ->with(['produk', 'mitra'])
                ->groupBy('stoks.produk_id', 'stoks.mitra_id', 'produks.minimum_stok')
                ->havingRaw('SUM(stoks.sisa) <= MIN(produks.minimum_stok)')
                ->get();

            $produkSales = Produk::whereHas('stoks', function($query) use ($userId) {
                $query->whereHas('mitra', function($q) use ($userId) {
                    $q->where('user_id', $userId);
                });
            })->get();

            $stokSales = Stok::with('mitra')
                ->whereHas('mitra', function ($q) use ($userId) {
                    $q->where('user_id', $userId);
                })->get();

            $labelMitraSales = [];
            $dataTerjualSales = [];

            foreach ($stokSales->groupBy('mitra.nama') as $namaMitra => $group) {
                $labelMitraSales[] = $namaMitra;
                $dataTerjualSales[] = $group->sum('terjual');
            }

            if (empty($labelMitraSales)) {
                $labelMitraSales[] = 'Tidak Ada';
                $dataTerjualSales[] = 0;
            }

            return view('dashboard.index', compact(
                'totalMitraSales',
                'stokMenipisSales',
                'produkSales',
                'labelMitraSales',
                'dataTerjualSales'
            ));
        }

        return redirect()->route('logout');
    }
}
